/*
 * Decompiled with CFR 0.152.
 */
package com.teamdev.jxbrowser.internal;

import com.teamdev.jxbrowser.internal.ChromiumExecutableLinux;
import com.teamdev.jxbrowser.internal.ChromiumExecutableMac;
import com.teamdev.jxbrowser.internal.ChromiumExecutableWin;
import com.teamdev.jxbrowser.logging.Logger;
import com.teamdev.jxbrowser.os.Environment;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Optional;
import java.util.stream.Stream;

public abstract class ChromiumExecutable {
    private static final String EXECUTABLE_NAME_FILE = "executable.name";
    protected final Path chromiumDir;
    private final String brandedName;
    private final Branding branding;

    public static ChromiumExecutable from(Path chromiumDir) {
        Optional<String> brandName = ChromiumExecutable.brandedExecutableName(chromiumDir);
        Branding branding = brandName.isPresent() ? Branding.CUSTOM : Branding.DEFAULT;
        String executableName = brandName.orElse(ChromiumExecutable.defaultName());
        if (Environment.isWindows()) {
            return new ChromiumExecutableWin(chromiumDir, executableName, branding);
        }
        if (Environment.isMac()) {
            return new ChromiumExecutableMac(chromiumDir, executableName, branding);
        }
        if (Environment.isLinux() && Environment.is64Bit()) {
            return new ChromiumExecutableLinux(chromiumDir, executableName, branding);
        }
        throw new IllegalStateException("Unsupported operating system.");
    }

    private static String defaultName() {
        if (Environment.isWindows()) {
            return "chromium";
        }
        if (Environment.isMac()) {
            return "Chromium";
        }
        if (Environment.isLinux()) {
            return "chromium";
        }
        throw new IllegalStateException("Unsupported operating system.");
    }

    ChromiumExecutable(Path chromiumDir, String name, Branding branding) {
        this.chromiumDir = chromiumDir;
        this.brandedName = name;
        this.branding = branding;
        if (ChromiumExecutable.containsPathSeparators(name)) {
            throw new IllegalArgumentException("Executable name must not contain path separators");
        }
    }

    public abstract Path path();

    public String name() {
        return this.brandedName;
    }

    public Branding branding() {
        return this.branding;
    }

    private static Optional<String> brandedExecutableName(Path chromiumDir) {
        Logger.debug("Searching for branded executable name in {0}", chromiumDir);
        if (!Files.exists(chromiumDir, new LinkOption[0])) {
            Logger.debug("Chromium directory does not exist: {0}", chromiumDir);
            return Optional.empty();
        }
        if (Environment.isMac()) {
            return ChromiumExecutable.findExecutableNameInAppBundle(chromiumDir);
        }
        return ChromiumExecutable.executableNameFromDir(chromiumDir);
    }

    private static Optional<String> executableNameFromDir(Path dir) {
        try {
            Path executableNameFile = dir.resolve(EXECUTABLE_NAME_FILE);
            if (!Files.exists(executableNameFile, new LinkOption[0])) {
                return Optional.empty();
            }
            return Optional.of(Files.readString(executableNameFile));
        }
        catch (IOException e) {
            Logger.error("Failed to read executable name", e);
            return Optional.empty();
        }
    }

    private static Optional<String> findExecutableNameInAppBundle(Path chromiumDir) {
        Optional<String> optional;
        block8: {
            Logger.debug("Searching for branded executable name in app bundle in {0}", chromiumDir);
            Stream<Path> chromiumDirEntries = Files.list(chromiumDir);
            try {
                optional = chromiumDirEntries.filter(bundle -> {
                    Path resourcesDir = bundle.resolve("Contents").resolve("Resources");
                    if (Files.exists(resourcesDir, new LinkOption[0])) {
                        String bundleName = ChromiumExecutable.appBundleName(bundle);
                        return ChromiumExecutable.executableNameFromDir(resourcesDir).map(name -> name.equals(bundleName)).orElse(false);
                    }
                    return false;
                }).findFirst().map(ChromiumExecutable::appBundleName);
                if (chromiumDirEntries == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (chromiumDirEntries != null) {
                        try {
                            chromiumDirEntries.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    Logger.error("Failed to find the executable name in app bundle", e);
                    return Optional.empty();
                }
            }
            chromiumDirEntries.close();
        }
        return optional;
    }

    private static String appBundleName(Path appBundleDir) {
        return appBundleDir.getFileName().toString().replace(".app", "");
    }

    private static boolean containsPathSeparators(String fileName) {
        return fileName.contains("\\") || fileName.contains("/");
    }

    public static enum Branding {
        CUSTOM,
        DEFAULT;

    }
}

