#!/usr/bin/env python3
#  Copyright (c) 2000-2025 TeamDev. All rights reserved.
#  TeamDev PROPRIETARY and CONFIDENTIAL.
#  Use is subject to license terms.

import argparse

from common import *
from repository.repository import repositories, ChromiumRoot
from update_icon import update_icon


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("-w", "--whitespace_action", type=str,
                        choices=['nowarn', 'warn', 'fix', 'error', 'error-all'],
                        default='fix',
                        help="Defines how Git should deal with patches "
                             "containing trailing whitespaces")
    parser.add_argument("-s", "--skip_verification",
                        action='store_true',
                        help="Skip Chromium version verification")
    parser.add_argument("-p", "--pre_sync",
                        action='store_true',
                        help="Apply pre-sync patches before calling gclient sync")
    parser.add_argument("-r", "--reset_patches",
                        action='store_true',
                        help="Reset all changes in all tracking repositories")
    args = parser.parse_args()
    current_chromium_version = ChromiumRoot().version()

    if not args.skip_verification and current_chromium_version != chromium_version:
        print("Version mismatch:")
        print("chromium/src:", current_chromium_version)
        print("this branch: ", chromium_version)
        sys.exit(1)

    if args.reset_patches:
        print("Resetting changes...")
        for repo in repositories:
            if not repo.reset_patches():
                print("Failed to reset patches.")
                sys.exit(1)
    else:
        if args.pre_sync:
            repo = ChromiumRoot()
            if not repo.apply_pre_sync_patches(args.whitespace_action):
                print("Failed to apply pre-sync patches.")
                sys.exit(1)
        else:
            if is_windows():
                update_icon()
            print("Processing repositories...")
            for repo in repositories:
                if not repo.apply_patches(args.whitespace_action):
                    print("Failed to apply patches.")
                    sys.exit(1)


if __name__ == "__main__":
    main()
