#  Copyright (c) 2000-2025 TeamDev. All rights reserved.
#  TeamDev PROPRIETARY and CONFIDENTIAL.
#  Use is subject to license terms.

import argparse
import stat

from common import *
from generate_chromium_info import generate_chromium_info_file


def get_7zr_app_name():
    if is_windows():
        return '7zr.exe'
    if is_linux():
        if is_arm():
            return '7zr-linux64-arm'
        return '7zr-linux64'
    if is_mac():
        return '7zr-mac'
    raise ValueError("Unsupported operating system or architecture.")


def compress(app, archive, files):
    if not run_command(os.getcwd(),
                       app,
                       'a',  # Stands for "archive" or "add"
                       '-snl',  # Store the hard links (directories) as links (required for macOS)
                       '-snh',  # Store the symbolic links (files) as links (required for macOS)
                       '-t7z',  # Type of archive 7z
                       '-mx5',  # Set the compression method to x5
                       archive,
                       *files):
        print('Files compression failed')
        sys.exit(1)


def set_execute_permission(app):
    if is_windows():
        return
    st = os.stat(app)
    os.chmod(app, st.st_mode | stat.S_IEXEC)


def main():
    parser = argparse.ArgumentParser("Compresses files using 7zr")
    parser.add_argument("-s", "--source_dir", type=str,
                        help="Path to the directory with the files to compress",
                        default="../target/bin")
    parser.add_argument("-t", "--target_dir", type=str,
                        help="Path to the directory where to put the archive",
                        default="../target")
    parser.add_argument("-c", "--target_cpu", type=str,
                        default=get_current_arch(),
                        help="The target CPU architecture.",
                        choices={"arm64", "x64", "x86"})
    parser.add_argument("-o", "--target_os", type=str,
                        default=get_current_os(),
                        help="The target operating system.",
                        choices={"mac", "win", "linux"})
    parser.add_argument("-n", "--name", type=str,
                        help="The name of the resulting file without an extension. Overrides target_cpu and target_os.")

    app = os.path.join(os.getcwd(), '7zr', get_7zr_app_name())
    set_execute_permission(app)

    files = []
    args = parser.parse_args()

    generate_chromium_info_file(args.source_dir, args.target_os, args.target_cpu, args.target_dir)

    fileset = os.listdir(args.source_dir)
    for file in fileset:
        files += [os.path.abspath(os.path.join(args.source_dir, file))]

    if args.name:
        archive_name = f"{args.name}.7z"
    else:
        arch_suffix = get_build_suffix(args.target_os, args.target_cpu)
        archive_name = f"chromium-{arch_suffix}.7z"
    archive_path = os.path.join(args.target_dir, archive_name)
    compress(app, archive_path, files)


if __name__ == "__main__":
    main()
