#  Copyright (c) 2000-2026 TeamDev. All rights reserved.
#  TeamDev PROPRIETARY and CONFIDENTIAL.
#  Use is subject to license terms.

import difflib
import os
import re
import sys

from common import run_command_with_output


class Diff:
    """
    A diff of a source file in the Chromium repository.
    """

    @staticmethod
    def from_patch_file(path):
        """
        Creates a new Diff instance from the given patch file.
        """
        patch_file = open(path, 'r', newline='\n')
        if not patch_file:
            raise Exception("Failed to open the patch file: " + path)
        diff = patch_file.read()
        return Diff(diff, path)

    def save_to_patch_file(self):
        """
        Saves the current diff to the corresponding patch file.
        """
        patch_dir = os.path.dirname(self.patch_file)
        if not os.path.exists(patch_dir):
            os.makedirs(patch_dir)
        f = open(self.patch_file, 'wb')
        f.truncate()
        f.write(self.diff.encode())
        f.close()

    def __init__(self, diff, patch_file):
        self.diff = diff
        self.patch_file = patch_file

    def __hash__(self):
        return hash(self.patch_file)

    def __ne__(self, other):
        return not self.__eq__(other)

    def __eq__(self, other):
        this_diff = self.diff.splitlines()
        other_diff = other.diff.splitlines()
        if len(this_diff) != len(other_diff):
            return False
        changed_lines = Diff.__changed_lines(difflib.ndiff(this_diff, other_diff))
        return len(changed_lines) == 0

    def __lt__(self, other):
        return self.patch_file < other.patch_file

    @staticmethod
    def __changed_lines(lines):
        return [line for line in lines if Diff.__is_different(line)]

    @staticmethod
    def __is_different(line):
        diff_symbols = ['+ ', '- ']
        if line[:2] in diff_symbols:
            is_index_line = line[2:].startswith('index')
            is_mark_line = line[2:].startswith('@@')
            return not is_index_line and not is_mark_line
        return False


class Diffs:

    @staticmethod
    def from_repository(repo):
        """
        Returns a list of Diff instances for the changed files in the given repository.
        """
        diffs = set()
        success, diff = run_command_with_output(repo.absolute_path, "git", 'diff', 'HEAD')
        if success:
            if len(diff) == 0:
                return diffs
            file_diffs = re.split(r'^(diff --git a/.* b/.*)$', diff, flags=re.MULTILINE)

            # The split will result in ["", header1, body1, header2, body2, ...], so we step by 2.
            for i in range(1, len(file_diffs), 2):
                header = file_diffs[i]
                body = file_diffs[i + 1]
                m = re.match(r'diff --git a/(.*) b/.*', header)
                if not m:
                    continue
                src_rel_path = m.group(1)

                # We replace the original Chromium icon, but we don't use git diffs for binary files.
                # So, we skip generating this patch, because we replace this file directly.
                if src_rel_path.endswith('.ico'):
                    continue

                # Antivirus software may identify some of Chromium test resource files as malware and delete them.
                # We're not supposed to delete Chromium files, so ignore such diffs.
                if re.search(r'^deleted file mode', body, re.MULTILINE):
                    print(f"Warning: Ignoring deleted file {src_rel_path}", file=sys.stderr)
                    continue
                patch_rel_path = src_rel_path + ".patch"
                diffs.add(Diff(header + body, os.path.abspath(os.path.join(repo.patches_dir, patch_rel_path))))
        return diffs

    @staticmethod
    def from_patches(repo):
        """
        Returns a list of Diff instances for the patch files from the `patches` directory
        for the given repository.
        """
        patches = set()
        for root, dirs, files in os.walk(repo.patches_dir):
            for patch_file in files:
                if os.path.basename(patch_file).endswith('.patch'):
                    patch_file_path = os.path.abspath(os.path.join(root, patch_file))
                    patches.add(Diff.from_patch_file(patch_file_path))
        return patches

    def __init__(self):
        pass
