#!/usr/bin/env python3
#  Copyright (c) 2000-2026 TeamDev. All rights reserved.
#  TeamDev PROPRIETARY and CONFIDENTIAL.
#  Use is subject to license terms.

import argparse

from project_config import *


def generate_project(args):
    config = ProjectConfig()

    # These settings are the same for all build configurations.
    config.dcheck_always_on(False)
    config.forbid_non_component_debug_builds(False)
    config.is_component_build(False)
    config.disable_fieldtrial_testing_config(True)
    config.use_qt5(False)
    config.use_qt6(False)
    # Proprietary features can be disabled at runtime.
    config.enable_proprietary_features()

    # Apply the config preset according to the build type.
    # Some settings may be overwritten by other parameters later.
    if args.dev:
        config.is_official_build(False)
        config.chrome_pgo_phase(0)
        config.allow_chromium_app_startup(True)
        config.enable_license_checker(False)
        config.treat_warnings_as_errors(False)
        config.blink_symbol_level(0)
        config.v8_symbol_level(0)
    if args.ci:
        config.is_official_build(False)
        config.chrome_pgo_phase(0)
        config.enable_license_checker(True)
        config.symbol_level(0)
        config.use_compiler_cache(True)
        config.treat_warnings_as_errors(True)
    if args.official:
        config.is_official_build(True)
        config.allow_chromium_app_startup(False)
        config.enable_license_checker(False)
        config.treat_warnings_as_errors(True)

    # Platform-specific settings are always the same.
    if is_windows():
        config.enable_precompiled_headers(False)

    if is_linux():
        if args.official:
            config.symbol_level(0)
        config.is_cfi(False)
        config.use_thin_lto(False)
        config.ozone_platform("x11")
        config.ozone_platform_x11(True)
        # Disabled explicitly because Chromium can pick the backend
        # during runtime, regardless of the previous setting.
        config.ozone_platform_wayland(False)
        if args.target_cpu == "arm64":
            config.use_xkbcommon(False)
    if is_mac():
        config.is_mac_os_native_chrome_icon_hidden(True)

    # Apply the remaining settings based on the passed parameters.
    config.target_cpu(args.target_cpu)
    config.build_jni_libraries(not args.no_jni_libraries)
    config.is_debug(args.debug)
    config.use_compiler_cache(args.cc)
    config.platinum_mas_build(args.platinum_mas_build)
    
    # Direct linkage is required for Platinum MAS builds.
    if (args.platinum_mas_build):
        config.enable_direct_linkage(True)
    if args.enable_license_checker:
        config.enable_license_checker(True)
    if args.vs:
        config.generate_vs_project()
    if args.qt:
        config.generate_qt_project()

    print("Generating a project with the following config:")
    print("=======================================")
    print(config)
    print("=======================================")
    if run_command(get_chromium_src_dir(), "gn", "gen", args.out_dir_name, *config.build()):
        return True
    else:
        print("Failed to generate the project.")
        return False


def main():
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)

    project_type = parser.add_mutually_exclusive_group(required=True)
    project_type.add_argument("--official", action="store_true",
                              help="A config to produce an official JxBrowser/DotNetBrowser build.")
    project_type.add_argument("--ci", action="store_true",
                              help="A config that is used on CI agents to ensure that the project is compilable.")
    project_type.add_argument("--dev", action="store_true",
                              help="A config for local dev builds.")

    parser.add_argument("--out_dir_name", type=str, metavar="", default="out/Release",
                        help="The build output directory.")
    parser.add_argument("--target_cpu", type=str, metavar="", default=get_current_arch(),
                        help="The target CPU architecture.",
                        choices={"arm64", "x64", "x86"})
    parser.add_argument("--enable_license_checker", action="store_true",
                        help="Enable license checking functionality.")
    parser.add_argument("--no_jni_libraries", action="store_true",
                        help="Do not build client libraries with JNI exports.")
    parser.add_argument("--debug", action="store_true",
                        help="Produce a debug build.")
    parser.add_argument("--cc", action="store_true",
                        help="Use the compiler cache.")
    parser.add_argument("--vs", action="store_true",
                        help="Generate Visual Studio solution files.")
    parser.add_argument("--qt", action="store_true",
                        help="Generate Qt project files.")
    parser.add_argument("--platinum_mas_build", action="store_true",
                        help="Produce a build compatible with Mac App Store.")

    args = parser.parse_args()

    generate_project(args)


if __name__ == "__main__":
    main()
