#  Copyright (c) 2000-2026 TeamDev. All rights reserved.
#  TeamDev PROPRIETARY and CONFIDENTIAL.
#  Use is subject to license terms.

import os
import platform
import shutil
import subprocess
import sys

from chromium_version import chromium_version

architectures = {'x86': '32', 'x64': '64', 'arm64': '_arm64'}
APP = 'Chromium.app'
VERSION = APP + '/Contents/Frameworks/Chromium\\ Framework.framework/Versions/{}'.format(chromium_version)
VERSION_RAW = APP + r'/Contents/Frameworks/Chromium Framework.framework/Versions/{}'.format(chromium_version)


def get_project_root():
    return os.path.join(get_chromium_src_dir(), "browsercore")


def get_chromium_src_dir():
    path = os.path.normpath(os.path.join(os.getcwd(), '..', '..'))
    if not is_chromium_src_dir(path):
        raise Exception("The project is not under the 'chromium/src' directory.")
    return path


def is_chromium_src_dir(path):
    if not os.path.exists(path):
        return False
    success, url = run_command_with_output(path, "git", "remote", "get-url", "origin")
    return success and url.startswith("https://chromium.googlesource.com/chromium/src.git")


def get_default_chromium_src_dir():
    path = os.path.normpath(os.path.join(os.getcwd(), '..', '..'))
    return path if is_chromium_src_dir(path) else None


def run_command_sysout(work_dir, *args):
    try:
        subprocess.check_call(
            args, stderr=sys.stderr, stdout=sys.stdout, cwd=work_dir, shell=is_windows())
    except subprocess.CalledProcessError as err:
        print(err.cmd)
        print(err.output)
        return False
    return True


def get_dynamic_libs_list(arch):
    arch_suffix = architectures[arch]
    if is_windows():
        return [
            "ipc" + arch_suffix + ".dll",
            "toolkit" + arch_suffix + ".dll",
            "awt_toolkit" + arch_suffix + ".dll",
        ]
    if is_linux():
        return [
            "libipc.so",
            "libtoolkit.so",
            "libawt_toolkit.so",
        ]
    if is_mac():
        return [
            VERSION_RAW + "/Libraries/libipc.dylib",
            VERSION_RAW + "/Libraries/libtoolkit.dylib",
            VERSION_RAW + "/Libraries/libawt_toolkit.dylib",
        ]


def run_command(work_dir, *args):
    try:
        output = subprocess.check_output(args, stderr=subprocess.STDOUT, cwd=work_dir,
                                         shell=is_windows())
        if output:
            print(output.decode("utf-8"))
    except subprocess.CalledProcessError as err:
        print(err.cmd)
        print(err.output)
        return False
    return True


def run_checked_command(work_dir, *args):
    print(args)
    if not run_command(work_dir, *args):
        raise Exception("Failed to execute the command.")


def run_shell_command(work_dir, args):
    try:
        output = subprocess.check_output(args, stderr=subprocess.STDOUT, cwd=work_dir, shell=True)
        return True, output.decode('utf-8')
    except subprocess.CalledProcessError as err:
        print(err.cmd)
        print(err.output)
        return False, err.output.decode('utf-8')


def run_command_with_output(work_dir, *args):
    try:
        output = subprocess.check_output(args, cwd=work_dir, shell=is_windows())
        return True, output.decode('utf-8')
    except subprocess.CalledProcessError as err:
        print(err.cmd)
        print(err.output)
        return False, None


def run_command_without_output(work_dir, *args):
    try:
        subprocess.check_output(args, stderr=subprocess.STDOUT, cwd=work_dir, shell=is_windows())
        return True
    except subprocess.CalledProcessError:
        return False


def is_windows():
    return platform.system() == 'Windows'


def is_linux():
    return platform.system() == 'Linux'


def is_arm():
    if is_mac():
        result, translated = run_command_with_output(os.getcwd(), "sysctl", "-n",
                                                     "sysctl.proc_translated")
        if result and int(translated):
            return True
    machine = platform.machine()
    return machine.startswith('arm') | machine.startswith("aarch64")


def is_mac():
    return platform.system() == 'Darwin'


def get_dest_file_path(src_dir, dest_dir, file_name):
    src_file = os.path.join(src_dir, file_name)
    base_name = os.path.basename(src_file)
    containing_dir = os.path.split(src_file)[0]
    if containing_dir == src_dir:
        return os.path.join(dest_dir, base_name)
    else:
        relative_path = os.path.relpath(containing_dir, src_dir)
        final_path = os.path.join(dest_dir, relative_path)
        if not os.path.exists(final_path):
            os.makedirs(final_path)
        return os.path.join(final_path, base_name)


def copy_files(src_dir, dest_dir, file_names):
    if not os.path.exists(dest_dir):
        os.makedirs(dest_dir)

    for file_name in file_names:
        src_file_path = os.path.join(src_dir, file_name)
        if os.path.exists(src_file_path):
            if os.path.isfile(src_file_path):
                dest_file_path = get_dest_file_path(src_dir, dest_dir, file_name)
                if not os.path.exists(dest_file_path):
                    shutil.copy(src_file_path, dest_file_path)
            else:
                dest_file_path = os.path.join(dest_dir, os.path.basename(src_file_path))
                # The destination directory must not already exist.
                if not os.path.exists(dest_file_path):
                    shutil.copytree(src_file_path, dest_file_path, True)


def copy_file(src_dir, dst_dir, src_file_name, dst_file_name):
    src_file = os.path.join(src_dir, src_file_name)
    dst_file = os.path.join(dst_dir, dst_file_name)
    shutil.copyfile(src_file, dst_file)


def remove_dir(path):
    if is_windows():
        run_command(os.getcwd(), 'rmdir', '/S', '/Q', path)
    else:
        shutil.rmtree(path, True)


def remove_existing_symlink(path):
    if os.path.islink(path):
        os.unlink(path)
    else:
        if os.path.isdir(path):
            remove_dir(path)
        else:
            os.remove(path)


def clear_dir_if_exists(dest_dir, only_dirs=False):
    if os.path.exists(dest_dir):
        if not only_dirs:
            shutil.rmtree(dest_dir)
        else:
            for _, dirs, _ in os.walk(dest_dir):
                for name in dirs:
                    remove_dir(os.path.join(dest_dir, name))


def get_current_chromium_version():
    result, output = run_shell_command(get_chromium_src_dir(), "git describe --tags")
    return output.strip()


def get_current_arch():
    if is_arm():
        return "arm64"
    arch = platform.machine()
    if arch.endswith("64"):
        return "x64"
    return "x86"


def get_current_os():
    """
    Returns the current operating system.
    """
    if is_linux():
        return "linux"
    elif is_windows():
        return "win"
    else:
        return "mac"


def get_build_suffix(operating_system, arch):
    """
    Returns a platform-specific suffix that we add to the names of binary archives.
    """
    if operating_system == "linux":
        return "linux64-arm" if arch == "arm64" else "linux64"
    if operating_system == "mac":
        return "mac-arm" if arch == "arm64" else "mac"
    if operating_system == "win":
        if arch == "arm64":
            return "win64-arm"
        return "win64" if arch == "x64" else "win32"
