#  Copyright (c) 2000-2026 TeamDev. All rights reserved.
#  TeamDev PROPRIETARY and CONFIDENTIAL.
#  Use is subject to license terms.

import os.path

from common import *
from . diff import Diffs


class Repository:
    """
    A repository in the Chromium source tree.
    """

    def __init__(self, rel_path):
        self.relative_path = rel_path
        self.absolute_path = os.path.join(get_chromium_src_dir(), rel_path)
        self.patches_dir = os.path.normpath(
            os.path.join(get_project_root(), "patches", self.name()))

    def generate_patches(self):
        """
        Generates patch files for the modified source files in this repository.
        """
        source_diffs = Diffs.from_repository(self)
        patch_diffs = Diffs.from_patches(self)

        new_patches = source_diffs.difference(patch_diffs)

        print("Repo:", self.name())
        for diff in sorted(new_patches):
            rel_path = os.path.relpath(diff.patch_file, self.patches_dir)
            print("\t" + rel_path)
            diff.save_to_patch_file()
            run_command(self.patches_dir, 'git', 'add', rel_path)
        if not new_patches:
            print("\tAll files are up-to-date.")

    def apply_patches(self, whitespace_action):
        """
        Applies patches to the source files in this repository, if any.
        """
        patch_diffs = Diffs.from_patches(self)
        source_diffs = Diffs.from_repository(self)

        print("Repo:", self.name())

        patches, outdated_sources = self.__collect_patches(source_diffs, patch_diffs)

        for source_file in sorted(outdated_sources):
            self._checkout_source_file(source_file)

        failed_patches = set()
        for patch in sorted(patches):
            if not self._apply_patch(patch, whitespace_action):
                failed_patches.add(patch)

        if not outdated_sources and not patches:
            print("\tAll files are up-to-date.")

        # Applying patches with 3-way merge takes significantly longer, so do it only
        # for the failed patches.
        if failed_patches:
            print("\n\tApply with 3-way merge [{}]:".format(len(failed_patches)))
            three_way_merge_patches = sorted(failed_patches)
            for patch in three_way_merge_patches:
                if self._apply_patch(patch, whitespace_action, True):
                    failed_patches.remove(patch)

        if failed_patches:
            print("\n\tFailed patches [{}]:".format(len(failed_patches)))
            for patch in sorted(failed_patches):
                print("\t\t" + os.path.relpath(patch, self.patches_dir))
            return False
        return True

    def reset_patches(self):
        print("Repo:", self.name())
        success, output = run_command_with_output(self.absolute_path, 'git', 'reset', '--hard')
        if success:
            print("\t" + output)
        else:
            print("\tFailed to reset patches.")
        return success

    def name(self):
        return self.relative_path

    def __collect_patches(self, source_diffs, patch_diffs):
        existing_files = set()
        for diff in source_diffs:
            existing_files.add(diff.patch_file)
        diffs = patch_diffs.difference(source_diffs)
        deleted_patches = source_diffs.difference(patch_diffs)
        outdated_sources = self.__collect_sources(diffs,
                                                  lambda patch: patch in existing_files)
        outdated_sources.update(self.__collect_sources(deleted_patches, lambda patch: True))
        new_patches = set(
            diff.patch_file for diff in diffs if self._can_apply_patch(diff.patch_file))
        return new_patches, outdated_sources

    def __collect_sources(self, diffs, condition):
        sources = set()
        for diff in diffs:
            if condition(diff.patch_file):
                patch_rel_path = os.path.relpath(diff.patch_file, self.patches_dir)
                source_file = patch_rel_path.replace(".patch", "")
                sources.add(source_file)
        return sources

    def _apply_patch(self, patch, whitespace_action, three_way_merge=False):
        rel_path = os.path.relpath(patch, self.patches_dir)

        cmd = ['git', 'apply', '--whitespace=' + whitespace_action]
        if three_way_merge:
            cmd.append('--3way')
        cmd.append(patch)
        success = run_command_without_output(self.absolute_path, *cmd)
        if success:
            print("{}{} {}{}{}".format('\t', 'git apply', rel_path, '\t', '[OK]'))
        else:
            print("{}{} {}{}{}".format('\t', 'git apply', rel_path, '\t', '[FAILED]'))

        # Unstage 3-way merged sources.
        if three_way_merge:
            source_file = rel_path.replace(".patch", "")
            run_command_without_output(self.absolute_path, 'git', 'restore',
                                       '--staged', source_file)
            # Reset conflicting changes.
            if not success:
                run_command_without_output(self.absolute_path, 'git', 'checkout', source_file)
        return success

    def _checkout_source_file(self, source_file):
        if run_command_without_output(self.absolute_path, 'git', 'checkout', source_file):
            print("{}{} {}{}{}".format('\t', 'git checkout', source_file, '\t', '[OK]'))
        else:
            print("{}{} {}{}{}".format('\t', 'git checkout', source_file, '\t', '[FAILED]'))

    def _can_apply_patch(self, patch):
        return True


class ChromiumRoot(Repository):
    """
    The root Chromium repository (chromium/src).
    """

    def __init__(self):
        Repository.__init__(self, ".")
        self.patches_dir = os.path.normpath(
            os.path.join(get_project_root(), "patches", self.name()))
        self.pre_sync_patches = []

    def apply_pre_sync_patches(self, whitespace_action='fix'):
        # The clang version in Chromium 102 causes a crash when an exception is thrown on ARM Macs.
        # Until this is fixed, we apply a set of patches that roll back clang to a previous version
        # that does not have this issue.
        print("Repo:", self.name())
        if is_mac() and is_arm():
            for patch in self.pre_sync_patches:
                source_file = patch.replace('.patch', '')
                if self.__is_modified(source_file):
                    self._checkout_source_file(source_file)
                if not self._apply_patch(os.path.join(self.patches_dir, patch), whitespace_action):
                    return False
        else:
            print("\tAll files are up-to-date.")
        return True

    def version(self):
        result, output = run_shell_command(self.absolute_path, "git describe --tags")
        return output.strip()

    def name(self):
        return "chromium/src"

    def _can_apply_patch(self, patch):
        # Pre-sync patches can only be applied explicitly to the chromium/src before gclient sync.
        for pre_sync_patch in self.pre_sync_patches:
            if patch.endswith(pre_sync_patch.replace('/', os.path.sep)):
                return False
        return True

    def __is_modified(self, source_file):
        return not run_command_without_output(self.absolute_path, 'git', 'diff', '--exit-code',
                                              source_file)


"""
A list of the repositories in which we track the modified source files.
"""
repositories = [
    ChromiumRoot(),
    Repository("third_party/devtools-frontend/src"),
    Repository("third_party/pdfium"),
]
