#!/usr/bin/env python3
# Copyright (c) 2000-2026 TeamDev. All rights reserved.
# TeamDev PROPRIETARY and CONFIDENTIAL.
# Use is subject to license terms.

"""
Delete Contents/Resources from every 'Chromium Helper*.app' inside Chromium.app.

Usage:
  python3 delete_unused_helper_resources.py [OUT_DIR]

Where OUT_DIR is a directory that contains Chromium.app.
Default OUT_DIR assumes you run the script from src/browsercore/tools and uses:
  ../../out/Release
"""

import argparse
import shutil
import sys
from pathlib import Path


def is_macos() -> bool:
    return sys.platform == "darwin"


def resolve_abs(path_str: str) -> Path:
    p = Path(path_str)
    try:
        return p.resolve(strict=False)
    except Exception:
        return p.absolute()


def find_helper_apps(chromium_app_dir: Path):
    # Finds directories like:
    # 'Chromium Helper.app', 'Chromium Helper (GPU).app', etc. inside Chromium.app.
    for app in chromium_app_dir.rglob("Chromium Helper*.app"):
        if app.is_dir():
            yield app


def delete_resources(app_dir: Path) -> bool:
    res = app_dir / "Contents" / "Resources"
    if not res.exists():
        print(f"No Resources dir in: {app_dir} (nothing to do)")
        return True

    try:
        shutil.rmtree(res)
        print(f"Deleted: {res}")
        return True
    except Exception as e:
        print(f"ERROR deleting {res}: {e}", file=sys.stderr)
        return False


def main(argv=None) -> int:
    if not is_macos():
        print("This cleanup is macOS-only; skipping.")
        return 0

    default_out_dir = Path("..") / ".." / "out" / "Release"

    parser = argparse.ArgumentParser(
        description="Delete Contents/Resources from Chromium Helper*.app bundles inside Chromium.app"
    )
    parser.add_argument(
        "out_dir",
        nargs="?",
        default=str(default_out_dir),
        help="Directory that contains Chromium.app (default: ../../out/Release when run from src/browsercore/tools)",
    )
    args = parser.parse_args(argv)

    out_dir = resolve_abs(args.out_dir)

    # Accept either a directory containing Chromium.app, or Chromium.app itself.
    if out_dir.name.endswith(".app"):
        chromium_app_dir = out_dir
    else:
        chromium_app_dir = out_dir / "Chromium.app"

    if not chromium_app_dir.exists() or not chromium_app_dir.is_dir():
        print(
            f"Chromium.app not found at: {chromium_app_dir}\n"
            f"Provided OUT_DIR: {out_dir}",
            file=sys.stderr,
        )
        return 1

    print(f"Scanning for 'Chromium Helper*.app' under: {chromium_app_dir}")

    apps = list(find_helper_apps(chromium_app_dir))
    if not apps:
        print("No helper app bundles found.")
        return 0

    errors = 0
    for app in apps:
        if not delete_resources(app):
            errors += 1

    print("Cleanup complete." if errors == 0 else f"Cleanup finished with {errors} error(s).")
    return 0 if errors == 0 else 1


if __name__ == "__main__":
    raise SystemExit(main())
